<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Lib\GetCommodity;
use App\Models\Commodity;
use Illuminate\Http\Request;
use App\Rules\FileTypeValidate;

class CommodityController extends Controller
{
    public function index()
    {
        $pageTitle   = 'Manage Commodity';
        $commodities = Commodity::searchable(['name'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.commodity.index', compact('pageTitle', 'commodities'));
    }

    public function create()
    {
        $pageTitle = 'New Commodity';
        return view('admin.commodity.edit', compact('pageTitle'));
    }

    public function save(Request $request, $id = 0)
    {
        $isRequired = $id ? 'nullable' : 'required';

        $request->validate([
            'name'                    => 'required',
            'unit'                    => 'required',
            'price'                   => 'required|numeric|gt:0',
            'market_capital'          => 'required|numeric|gt:0',
            'twenty_four_hour_change' => 'required|numeric',
            'seven_day_change'        => 'required|numeric',
            'image'                   => [$isRequired, 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])]
        ]);

        if ($id) {
            $commodity = Commodity::findOrFail($id);
            $message   = "Commodity updated successfully";
        } else {
            $commodity = new Commodity();
            $message   = "Commodity added successfully";
        }

        if ($request->hasFile('image')) {
            $commodity->image = fileUploader($request->image, getFilePath('commodityImage'), getFileSize('commodityImage'), @$commodity->image);
        }

        $commodity->name                    = $request->name;
        $commodity->unit                    = $request->unit;
        $commodity->price                   = $request->price;
        $commodity->market_capital          = $request->market_capital;
        $commodity->twenty_four_hour_change = $request->twenty_four_hour_change;
        $commodity->seven_day_change        = $request->seven_day_change;
        $commodity->save();

        $notify[] = ['success', $message];
        return back()->withNotify($notify);
    }

    public function edit($id)
    {
        $pageTitle = 'Edit Commodity';
        $commodity = Commodity::findOrFail($id);
        return view('admin.commodity.edit', compact('pageTitle', 'commodity'));
    }


    public function status($id)
    {
        return Commodity::changeStatus($id);
    }

    public function fetchCommodities()
    {
        $pageTitle = "Fetch Commodities";

        $apiResponse = GetCommodity::getApiCommodities();
        if (!$apiResponse['success']) {
            $notify[] = ['error', $apiResponse['message']];
            return back()->withNotify($notify);
        }

        $apiCommodities = $apiResponse['data'];

        $existingCommodities = Commodity::pluck('name')->toArray();

        $newCommodities = array_filter($apiCommodities, function ($commodity) use ($existingCommodities) {
            return !in_array($commodity['name'], $existingCommodities);
        });

        return view('admin.commodity.fetch', compact('pageTitle', 'newCommodities'));
    }

    public function storeFetched(Request $request)
    {
        $request->validate([
            'commodities' => 'required|array',
            'commodities.*' => 'required|string'
        ]);

        $selectedCommodities = $request->commodities;

        $apiResponse = GetCommodity::getApiCommodities();

        if (!$apiResponse['success']) {
            $notify[] = ['error', $apiResponse['message']];
            return back()->withNotify($notify);
        }

        $apiCommodities = $apiResponse['data'];

        if (!$apiCommodities) {
            $notify[] = ['error', 'Failed to fetch commodities from API'];
            return back()->withNotify($notify);
        }

        $storedCount = 0;

        foreach ($apiCommodities as $apiCommodity) {
            if (in_array($apiCommodity['name'], $selectedCommodities)) {
                $exists = Commodity::where('name', $apiCommodity['name'])->exists();

                if (!$exists) {

                    $unit = $apiCommodity['unit'];
                    if (strpos($unit, '/') !== false) {
                        $unit = substr($unit, strpos($unit, '/') + 1);
                    }

                    $commodity = new Commodity();
                    $commodity->name                    = $apiCommodity['name'];
                    $commodity->unit                    = $unit;
                    $commodity->price                   = $apiCommodity['price'];
                    $commodity->market_capital          = 0;
                    $commodity->twenty_four_hour_change = 0;
                    $commodity->seven_day_change        = 0;
                    $commodity->save();

                    $storedCount++;
                }
            }
        }

        $notify[] = ['success', $storedCount . ' commodities stored successfully'];
        return redirect()->route('admin.commodity.index')->withNotify($notify);
    }



}
