<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Lib\CurlRequest;
use App\Lib\GetCommodity;
use App\Models\Commodity;
use App\Models\CommodityPriceSnapshot;
use App\Models\CronJob;
use App\Models\CronJobLog;
use Carbon\Carbon;

class CronController extends Controller
{
    public function cron()
    {
        $general            = gs();
        $general->last_cron = now();
        $general->save();

        $crons = CronJob::with('schedule');

        if (request()->alias) {
            $crons->where('alias', request()->alias);
        } else {
            $crons->where('next_run', '<', now())->where('is_running', Status::YES);
        }
        $crons = $crons->get();
        foreach ($crons as $cron) {
            $cronLog              = new CronJobLog();
            $cronLog->cron_job_id = $cron->id;
            $cronLog->start_at    = now();
            if ($cron->is_default) {
                $controller = new $cron->action[0];
                try {
                    $method = $cron->action[1];
                    $controller->$method();
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            } else {
                try {
                    CurlRequest::curlContent($cron->url);
                } catch (\Exception $e) {
                    $cronLog->error = $e->getMessage();
                }
            }
            $cron->last_run = now();
            $cron->next_run = now()->addSeconds($cron->schedule->interval);
            $cron->save();

            $cronLog->end_at = $cron->last_run;

            $startTime         = Carbon::parse($cronLog->start_at);
            $endTime           = Carbon::parse($cronLog->end_at);
            $diffInSeconds     = $startTime->diffInSeconds($endTime);
            $cronLog->duration = $diffInSeconds;
            $cronLog->save();
        }
        if (request()->target == 'all') {
            $notify[] = ['success', 'Cron executed successfully'];
            return back()->withNotify($notify);
        }
        if (request()->alias) {
            $notify[] = ['success', keyToTitle(request()->alias) . ' executed successfully'];
            return back()->withNotify($notify);
        }
    }

    public function priceUpdate()
    {
        $commodities = Commodity::active()->with('priceSnapshot')->get();

        $apiResponse = GetCommodity::getApiCommodities();
        if (!$apiResponse['success']) {
            $notify[] = ['error', $apiResponse['message']];
            return back()->withNotify($notify);
        }

        $apiCommodities = collect($apiResponse['data']);
        $updatedCount = 0;
        $weeklyUpdatedCount = 0;
        $now = Carbon::now();

        foreach ($commodities as $commodity) {
            $match = $apiCommodities->firstWhere('name', $commodity->name);
            if ($match && isset($match['price'])) {
                $newPrice = $match['price'];

                if ($commodity->priceSnapshot) {

                    $snapshot = $commodity->priceSnapshot;
                } else {

                    $snapshot = new CommodityPriceSnapshot();
                    $snapshot->commodity_id = $commodity->id;
                    $snapshot->price_24h_ago = $commodity->price;
                    $snapshot->price_7d_ago = $commodity->price;
                    $snapshot->last_24h_update = $now;
                    $snapshot->last_7d_update = $now;
                    $snapshot->save();
                }

                $shouldUpdate7Day = false;
                if (!$snapshot->last_7d_update || $snapshot->last_7d_update->diffInDays($now) >= 7) {
                    $snapshot->price_7d_ago = $commodity->price;
                    $snapshot->last_7d_update = $now;
                    $shouldUpdate7Day = true;
                    $weeklyUpdatedCount++;
                }

                $snapshot->price_24h_ago = $commodity->price;
                $snapshot->last_24h_update = $now;

                $commodity->price = $newPrice;

                $commodity->twenty_four_hour_change = $this->calculateChange($newPrice, $snapshot->price_24h_ago);
                $commodity->seven_day_change = $this->calculateChange($newPrice, $snapshot->price_7d_ago);

                $snapshot->save();
                $commodity->save();

                $updatedCount++;
            }
        }

        $message = "Price updated successfully. {$updatedCount} commodities updated.";
        if ($weeklyUpdatedCount > 0) {
            $message .= " {$weeklyUpdatedCount} weekly snapshots updated.";
        }

        $notify[] = ['success', $message];
        return back()->withNotify($notify);
    }

    private function calculateChange($currentPrice, $oldPrice)
    {
        if ($oldPrice == 0) {
            return 0.00;
        }

        return round((($currentPrice - $oldPrice) / $oldPrice) * 100, 2);
    }
}
