<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\Commodity;
use App\Models\Trade;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TradeController extends Controller
{
    public function index()
    {
        $pageTitle   = 'New Trade';
        $commodities = Commodity::active()->hasPrice()->orderBy('name')->get();
        return view('Template::user.trade.index', compact('pageTitle', 'commodities'));
    }

    public function history()
    {
        $pageTitle = 'Trade History';
        $trades    = Trade::where('user_id', auth()->id())->searchable(['commodity:name'])->with('commodity')->latest()->paginate(getPaginate());
        return view('Template::user.trade.history', compact('pageTitle', 'trades'));
    }

    public function getAssetBalance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commodity' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->all()
            ]);
        }
        $commodity = Commodity::where('id', $request->commodity)->first();

        if (!$commodity) {
            return response()->json([
                'success' => false,
                'message' => "Commodity not found"
            ]);
        }

        $asset = Asset::where('user_id', auth()->id())->where('commodity_id', $request->commodity)->first();

        return response()->json([
            'balance' => showAmount(@$asset->quantity ?? 0, currencyFormat: false) . ' ' . $commodity->unit,
            'success' => true
        ]);
    }

    public function trade(Request $request)
    {

        $request->validate([
            'amount'    => "required|numeric|gt:0",
            'commodity' => "required|numeric|gt:0",
            'type'      => 'required|numeric|in:' . Status::TRADE_TYPE_BUY . ',' . Status::TRADE_TYPE_SELL,
        ]);

        $user      = auth()->user();
        $commodity = Commodity::active()->hasPrice()->findOrFail($request->commodity);
        $asset     = $user->assets->where('commodity_id', $commodity->id)->first();

        if (!$asset) {
            $asset               = new Asset();
            $asset->user_id      = $user->id;
            $asset->commodity_id = $commodity->id;
            $asset->save();
        }

        $qty         = $request->amount / $commodity->price;
        $totalAmount = $qty * $commodity->price;

        $trx                = getTrx();
        $asset->price       = $commodity->price;
        $asset->total_cost += $totalAmount;



        if ($request->type == Status::TRADE_TYPE_BUY) {

            if ($totalAmount > $user->balance) {
                $notify[] = ['error', 'Insufficient balance'];
                return back()->withNotify($notify);
            }

            $user->balance -= $totalAmount;
            $user->save();

            $details = 'Balance subtract for buy commodity(' . $commodity->name . '-' . showAmount($qty, currencyFormat: false) . ' ' . $commodity->unit . ')';
            $this->createTrx("-", $totalAmount, $user, "buy_commodity", $details, $trx);

            $asset->quantity += $qty;
            $asset->save();

            $details = "Buy commodity - {$commodity->name}";
            $this->createTrx("+", $qty, $user, "buy_commodity", $details, $trx, $commodity, $asset);

            $tradeType = 'Buy';
            $act       = 'COMMODITY_BUY';
        } else {

            if ($asset->quantity < $qty) {
                $notify[] = ['error', 'Insufficient asset quantity'];
                return back()->withNotify($notify);
            }

            $asset->quantity -= $qty;
            $asset->save();

            $details = "Sale commodity - {$commodity->name}";
            $this->createTrx("-", $qty, $user, "sale_commodity", $details, $trx, $commodity, $asset);

            $user->balance += $request->amount;
            $user->save();

            $details = 'Balance added for sale commodity(' . $commodity->name . '-' . showAmount($qty, currencyFormat: false) . ' ' . $commodity->unit . ')';
            $this->createTrx("+", $totalAmount, $user, "sale_commodity", $details, $trx);

            $tradeType = 'Sale';
            $act       = 'COMMODITY_SALE';
        }

        $trade               = new Trade();
        $trade->user_id      = $user->id;
        $trade->commodity_id = $commodity->id;
        $trade->type         = $request->type;
        $trade->quantity     = $qty;
        $trade->price        = $commodity->price;
        $trade->total_cost   = $request->amount;
        $trade->save();

        notify($user, $act, [
            'commodity'     => $commodity->name,
            'price'         => showAmount($commodity->price, currencyFormat: false),
            'unit'          => $commodity->unit,
            'quantity'      => showAmount($trade->quantity, 8, currencyFormat: false),
            'total_amount'  => showAmount($request->amount, currencyFormat: false),
            'asset_balance' => showAmount($asset->quantity, 8, currencyFormat: false),
            'main_balance'  => showAmount($user->balance, currencyFormat: false),
            'trx'           => $trx,
        ]);

        $notify[] = ['success', 'Commodity ' . $tradeType . ' successfully'];
        return back()->withNotify($notify);
    }

    private function createTrx($trxType, $amount, $user, $remark, $detail, $trx, $commodity = null, $asset = null)
    {
        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->commodity_id = @$commodity->id ?? 0;
        $transaction->amount       = $amount;
        if ($commodity) {
            $transaction->post_balance = $asset->quantity;
        } else {
            $transaction->post_balance = $user->balance;
        }
        $transaction->charge       = 0;
        $transaction->trx_type     = $trxType;
        $transaction->details      = $detail;
        $transaction->trx          = $trx;
        $transaction->remark       = $remark;
        $transaction->save();
    }
}
